/**
 * OWASP AppSensor
 * 
 * This file is part of the Open Web Application Security Project (OWASP)
 * AppSensor project. For details, please see
 * <a href="http://www.owasp.org/index.php/Category:OWASP_AppSensor_Project">
 * 	http://www.owasp.org/index.php/Category:OWASP_AppSensor_Project</a>.
 *
 * Copyright (c) 2010 - The OWASP Foundation
 * 
 * AppSensor is published by OWASP under the BSD license. You should read and accept the
 * LICENSE before you use, modify, and/or redistribute this software.
 * 
 * @author Michael Coates <a href="http://www.aspectsecurity.com">Aspect Security</a>
 * @author John Melton <a href="http://www.jtmelton.com/">jtmelton</a>
 * @created 2010
 */
namespace org.owasp.appsensor
{

    using System;
    using org.owasp.appsensor.intrusiondetection;
    using org.owasp.appsensor.trendmonitoring;

    /**
     * AppSensor locator class is provided to make it easy to gain access to the 
     * current AppSensor classes in use. Use the set methods to override the reference 
     * implementations with instances of any custom implementations.  Alternatively, 
     * These configurations are set in the appsensor.properties file.
     * 
     * 9/13/2010 - jtm - Fix for issue http://code.google.com/p/appsensor/issues/detail?id=3
     * Needed to update the APPSENSOR class to match the way Esapi's ObjFactory class now does 
     * object creation since this is a dependency.  
     * 
     * 9/22/2010 - jtm - Updated class to use ASObjFactory to remove Esapi dependency.
     * 
     * @author Michael Coates (michael.coates .at. owasp.org) 
     *         <a href="http://www.aspectsecurity.com">Aspect Security</a>
     * @author John Melton (jtmelton .at. gmail.com)
     *         <a href="http://www.jtmelton.com/">jtmelton</a>
     * @since May 6, 2010
     */
    public class APPSENSOR
    {

        private static object accessorLock = new object();

        /** Intrusion storage */
        private static IntrusionStore intrusionStore = null;

        /** Response handler */
        private static ResponseAction responseAction = null;

        /** Utilities object */
        private static ASUtilities asUtilities = null;

        /** Trend Logger */
        private static TrendLogger trendLogger = null;

        /** Implementation of security config that pulls appropriate fields from appsensor.properties */
        private static AppSensorSecurityConfiguration assc = (AppSensorSecurityConfiguration)AppSensorSecurityConfiguration.GetInstance();

        /**
         * prevent instantiation of this class
         */
        private APPSENSOR() { }


        /**
         * @return the current APPSENSOR IntrusionStore object being used 
         * for this application. 
         */
        public static IntrusionStore IntrusionStore
        {
            get
            {
                return ASObjFactory.Make<IntrusionStore>(assc.IntrusionStore, "IntrusionStore");
            }
        }

        /**
         * Change the current APPSENSOR IntrusionStore object to the IntrusionStore object provided. 
         * @param intrusionStore
         *            the IntrusionStore object to set to be the current APPSENSOR IntrusionStore object.
         * @return
         * 		The previous APPSENSOR IntrusionStore, may be null            
         */
        public static IntrusionStore SetIntrusionStore(IntrusionStore intrusionStore)
        {
            IntrusionStore previousIntrusionStore;
            lock (accessorLock)
            {
                previousIntrusionStore = APPSENSOR.intrusionStore;
                APPSENSOR.intrusionStore = intrusionStore;
            }
            return previousIntrusionStore;
        }

        /**
         * @return the current APPSENSOR ServiceStore object being used 
         * for this application. 
         */
        public static ServiceStore ServiceStore
        {
            get
            {
                return ASObjFactory.Make<ServiceStore>(assc.ServiceStore, "ServiceStore");
            }
        }

        /**
         * @return the current APPSENSOR ResponseAction object being used
         * for this application. 
         */
        public static ResponseAction ResponseAction
        {
            get
            {
                return ASObjFactory.Make<ResponseAction>(assc.ResponseAction, "ResponseAction");
            }
        }

        /**
         * Change the current APPSENSOR ResponseAction object to the ResponseAction object provided. 
         * @param responseAction
         *            the ResponseAction object to set to be the current APPSENSOR ResponseAction object.
         * @return
         * 		The previous APPSENSOR ResponseAction, may be null            
         */
        public static ResponseAction SetResponseAction(ResponseAction responseAction)
        {
            ResponseAction previousResponseAction;
            lock (accessorLock)
            {
                previousResponseAction = APPSENSOR.responseAction;
                APPSENSOR.responseAction = responseAction;
            }
            return previousResponseAction;
        }

        /**
         * @return the current APPSENSOR ASUtilities object being used
         * for this application. 
         */
        public static ASUtilities AsUtilities
        {
            get
            {
                return ASObjFactory.Make<ASUtilities>(assc.AsUtilities, "ASUtilities");
            }
        }

        /**
         * Change the current APPSENSOR ASUtilities object to the ASUtilities object provided. 
         * @param asUtilities
         *            the ASUtilities object to set to be the current APPSENSOR ASUtilities object.
         * @return
         * 		The previous APPSENSOR ASUtilities, may be null            
         */
        public static ASUtilities setASUtilities(ASUtilities asUtilities)
        {
            ASUtilities previousASUtilities;
            lock (accessorLock)
            {
                previousASUtilities = APPSENSOR.asUtilities;
                APPSENSOR.asUtilities = asUtilities;
            }
            return previousASUtilities;
        }

        /**
         * @return the current APPSENSOR TrendLogger object being used
         * for this application. 
         */
        public static TrendLogger TrendLogger
        {
            get
            {
                return ASObjFactory.Make<TrendLogger>(assc.TrendLogger, "TrendLogger");
            }
        }

        /**
         * Change the current APPSENSOR TrendLogger object to the TrendLogger object provided. 
         * @param trendLogger
         *            the TrendLogger object to set to be the current APPSENSOR TrendLogger object.
         * @return
         * 		The previous APPSENSOR TrendLogger, may be null            
         */
        public static TrendLogger SetTrendLogger(TrendLogger trendLogger)
        {
            TrendLogger previousTrendLogger;
            lock (accessorLock)
            {
                previousTrendLogger = APPSENSOR.trendLogger;
                APPSENSOR.trendLogger = trendLogger;
            }
            return previousTrendLogger;
        }
    }
}